#!/bin/bash
set -e

GREEN='\033[0;32m'
RED='\033[0;31m'
NC='\033[0m' # No Color

echo -e "${GREEN}Installing LPR License Service...${NC}"

if [ "$EUID" -ne 0 ]; then
  echo -e "${RED}Please run as root (sudo)${NC}"
  exit 1
fi

INSTALL_DIR="/opt/license-service"
BIN_DIR="$INSTALL_DIR/bin"
SYSTEMD_DIR="/etc/systemd/system"

echo "Creating directory..."
mkdir -p "$BIN_DIR"

echo "Copying executable file..."
cp "$PWD/lpr-scripts/license_service" "$BIN_DIR/"
chmod +x "$BIN_DIR/license_service"

echo "Creating config file..."
cat > "$INSTALL_DIR/config.json" << EOF
{
  "port": 8095,
  "host": "127.0.0.1",
  "secretKey": "$(openssl rand -hex 16)",
  "heartbeatTimeoutSecs": 30
}
EOF

echo "Creating license-service user..."
id -u license-service &>/dev/null || useradd -r -s /bin/false -M license-service

echo "Setting up permissions..."
chown -R license-service:license-service "$INSTALL_DIR"
chmod 640 "$INSTALL_DIR/config.json"

echo "Creating systemd service file..."
cat > "$SYSTEMD_DIR/license-service.service" << EOF
[Unit]
Description=License Service
After=network.target

[Service]
Type=simple
User=license-service
WorkingDirectory=$INSTALL_DIR
ExecStart=$BIN_DIR/license_service
Restart=on-failure
RestartSec=5
StandardOutput=syslog
StandardError=syslog
SyslogIdentifier=license-service

PrivateTmp=true
ProtectSystem=full
NoNewPrivileges=true

[Install]
WantedBy=multi-user.target
EOF

echo "Restarting systemd..."
systemctl daemon-reload

echo "Launching service..."
systemctl enable license-service.service
systemctl start license-service.service

echo "Checking service status..."
systemctl status license-service.service --no-pager

echo -e "${GREEN}License Service installation finished successfully!${NC}"
